library IEEE;
use IEEE.std_logic_1164.ALL;
use IEEE.std_logic_signed.ALL;
library unisim; 
use unisim.vcomponents.all; 

entity USB_AUDIO is
port (
   RESET                 : in std_logic;
   SCL                   : in std_logic;
   SDA                   : inout std_logic;
   IFCLK                 : in std_logic;
   FX2_FLAGA             : in std_logic;
   DET                   : in std_logic;	-- VBUS DETECT
   CLK22                 : in std_logic;
   CLK24                 : in std_logic;
   CLK24O                : out std_logic;
   SCLK_P                : out std_logic;
   BCLK_P                : out std_logic;
   LRCK_P                : out std_logic;
   SDATA_P               : out std_logic;
   SCLK_N                : out std_logic;
   BCLK_N                : out std_logic;
   LRCK_N                : out std_logic;
   SDATA_N               : out std_logic;
   MCLKI                 : in std_logic;	-- AKM in
   BCLKI                 : in std_logic;	-- AKM in
   LRCKI                 : in std_logic;	-- AKM in
   SDATAI                : in std_logic;	-- AKM in
   CPU_INT               : out std_logic;
   CPU_SCL               : in std_logic;
   CPU_SDA               : inout std_logic;
   CS_N                  : out std_logic;	-- AKM cont
   CCLK                  : out std_logic;	-- AKM cont
   CDTI                  : out std_logic;	-- AKM cont
   CDTO                  : in std_logic;	-- AKM cont
   INT0                  : in std_logic;	-- AKM cont
   INT1                  : in std_logic;	-- AKM cont
----------------------------------------
   FX2_FD                : in std_logic_vector(7 downto 0);
   FX2_SLWR              : out std_logic;
   FX2_SLOE              : out std_logic;
   FX2_SLRD              : out std_logic;
   FX2_PKTEND            : out std_logic;
   FX2_FIFOADR           : out std_logic_vector(1 downto 0)
   );
end;

architecture rtl of USB_AUDIO is

   signal    count8             : std_logic_vector(7 downto 0);
   signal    cnt_ad             : std_logic_vector(2 downto 0);

   signal    count              : std_logic_vector(8 downto 0);
   signal    spd_para_dt        : std_logic_vector(23 downto 0);

   signal    pcm_datal          : std_logic_vector(23 downto 0);
   signal    pcm_datar          : std_logic_vector(23 downto 0);

   signal    DATAOUT            : std_logic_vector (7 downto 0);
   signal    OUTLAT             : std_logic;

   signal    file_mode          : std_logic_vector (2 downto 0);
   signal    file_mode_d        : std_logic_vector (2 downto 0);
   signal    file_mode_2d       : std_logic_vector (2 downto 0);

--   signal    RJ24_H             : std_logic;
   signal    XIN                : std_logic;
   signal    XTLCLK             : std_logic;
   signal    clk_sel            : std_logic;
   signal    usb_bclk           : std_logic;
   signal    usb_lrck           : std_logic;
   signal    usb_sdata          : std_logic;

   signal    cs_n_cpu           : std_logic;
   signal    cclk_cpu           : std_logic;
   signal    cdti_cpu           : std_logic;
   signal    if_cnt             : std_logic_vector(2 downto 0);
   signal    slrd_uaa           : std_logic;
   signal    ram_re_mask        : std_logic;

   signal    flaga_d            : std_logic;
   signal    ram_we             : std_logic;
   signal    ram_re             : std_logic;

   signal    add_mark_flg       : std_logic;
   signal    mark_fd            : std_logic;
   signal    mark               : std_logic;
   signal    det_mark_err       : std_logic;
   signal    ram_wr_addr_reg    : std_logic_vector(7 downto 0);
   signal    ram_waddr_reg_val  : std_logic;

   signal    pb_data            : std_logic_vector(7 downto 0);

   signal    full               : std_logic;
   signal    empty              : std_logic;
   signal    overflow           : std_logic;
   signal    underflow          : std_logic;
   signal    rd_data_count      : std_logic_vector(10 downto 0);
   signal    wr_data_count      : std_logic_vector(10 downto 0);
   signal    wr_rst             : std_logic;
   signal    rd_rst             : std_logic;

   signal    rd_rst_d           : std_logic;
   signal    rd_rst_2d          : std_logic;
   signal    rd_rst_3d          : std_logic;

   signal    validity           : std_logic;
   signal    user               : std_logic;
   signal    channel            : std_logic;
   signal    temp               : std_logic;
   signal    parity             : std_logic;
   signal    frmcnt             : std_logic_vector(7 downto 0);
   signal    spdata             : std_logic;

   signal   cpu_scli            : std_logic;
   signal   cpu_sdai            : std_logic;
   signal   cpu_sdao            : std_logic;
   signal   fpga_irq            : std_logic;
   signal   is_usb_h            : std_logic;

   signal   sdai      : std_logic;
   signal   sdao      : std_logic;

   signal   SCLK      : std_logic;
   signal   BCLK      : std_logic;
   signal   LRCK      : std_logic;
   signal   SDATA     : std_logic;

component I2CIF
   port (
    RESET       : in std_logic;
    SCLK        : in std_logic;
    SCL         : in std_logic;
    SDAI        : in std_logic;
    SDAO        : out std_logic;
    RDBK        : in std_logic_vector (7 downto 0);
    DATA        : out std_logic_vector (7 downto 0);
    WE          : out std_logic
    );
end component ;

component fifo_generator_v8_2
	port (
	wr_clk: in std_logic;
	wr_rst: in std_logic;
	rd_clk: in std_logic;
	rd_rst: in std_logic;
	din: in std_logic_vector(8 downto 0);
	wr_en: in std_logic;
	rd_en: in std_logic;
	dout: out std_logic_vector(8 downto 0);
	full: out std_logic;
	overflow: out std_logic;
	empty: out std_logic;
	underflow: out std_logic;
	rd_data_count: out std_logic_vector(10 downto 0);
	wr_data_count: out std_logic_vector(10 downto 0)
  );
end component ;

component I2C_SLAVE
   port (
    RESET       : in std_logic;
    SCLK        : in std_logic;
    SCL         : in std_logic;
    SDAI        : in std_logic;
    SDAO        : out std_logic;
    CS_N        : out std_logic;
    CCLK        : out std_logic;
    CDTI        : out std_logic;
    CDTO        : in std_logic;
    FPGA_IRQ    : out std_logic;
    USB_H       : in std_logic;
    FILE_MODE   : in std_logic_vector(2 downto 0)
    );
end component ;

component OBUFDS
   PORT (
   I    : IN STD_LOGIC;
   O    : OUT STD_LOGIC; 
   OB   : OUT STD_LOGIC 
   );
end component ;

component clkctrl
   PORT ( 
   clkselect  : IN  STD_LOGIC;
   inclk0x    : IN  STD_LOGIC;
   inclk1x    : IN  STD_LOGIC;
   outclk     : OUT  STD_LOGIC
   ); 
end component ;

begin

   u0:I2CIF port map (
   RESET => RESET,
   SCLK => IFCLK,
   SCL => SCL,
   SDAI => sdai,
   SDAO => sdao,
   RDBK => ram_wr_addr_reg,
   DATA => DATAOUT,
   WE => OUTLAT
   );

   u1:fifo_generator_v8_2 port map (
	wr_clk => IFCLK,
        wr_rst => wr_rst,
	rd_clk => XIN,
        rd_rst => rd_rst,
	din(8 downto 1) => FX2_FD,
        din(0) => mark_fd,
	wr_en => ram_we,
	rd_en => ram_re,
	dout(8 downto 1) => pb_data,
        dout(0) => mark,
	full => open,
	overflow => overflow,
	empty => open,
	underflow => underflow,
	rd_data_count => rd_data_count,
	wr_data_count => wr_data_count
   );

   u2:I2C_SLAVE port map (
    RESET => RESET,
    SCLK => CLK22,
    SCL => CPU_SCL,
    SDAI => cpu_sdai,
    SDAO => cpu_sdao,
    CS_N => CS_N,
    CCLK => CCLK,
    CDTI => CDTI,
    CDTO => CDTO,
    FPGA_IRQ => fpga_irq,
    USB_H => DET,
    FILE_MODE => file_mode
    );

   u3a:clkctrl port map (
   clkselect => clk_sel,
   inclk0x => CLK22,
   inclk1x => CLK24,
   outclk => XTLCLK
   );

   u3b:BUFGMUX port map (
   I0 => MCLKI,
   I1 => XTLCLK,
   S => is_usb_h,
   O => XIN
   );

   u4:OBUFDS port map (
   I => SCLK,
   O => SCLK_P,
   OB => SCLK_N
   );

   u5:OBUFDS port map (
   I => BCLK,
   O => BCLK_P,
   OB => BCLK_N
   );

   u6:OBUFDS port map (
   I => LRCK,
   O => LRCK_P,
   OB => LRCK_N
   );

   u7:OBUFDS port map (
   I => SDATA,
   O => SDATA_P,
   OB => SDATA_N
   );

   SDA <= '0' when sdao = '0' else 'Z';
   sdai <= '1' when SDA /= '0' else '0';

   CPU_INT <= INT0 or INT1 or fpga_irq;

   CPU_SDA <= '0' when cpu_sdao = '0' else 'Z';
   cpu_sdai <= '1' when CPU_SDA /= '0' else '0';

   clk_sel <= '0' when ( file_mode(2 downto 1) = "11" or file_mode(0) = '0' ) else '1';	--22 at '0'

   FX2_PKTEND <= '1';
   FX2_SLOE <= '0';
   FX2_FIFOADR <= "00";
   FX2_SLRD <= slrd_uaa;
   FX2_SLWR <= '1';

   is_usb_h <= DET;
   SCLK <= XTLCLK when is_usb_h = '1' else MCLKI;
   BCLK <= usb_bclk when is_usb_h = '1' else BCLKI;
   LRCK <= usb_lrck when is_usb_h = '1' else LRCKI;
   SDATA <= usb_sdata when is_usb_h = '1' else SDATAI;

   CLK24O <= CLK24;

   process(IFCLK)
   begin
      if (IFCLK'event and IFCLK = '1') then
         flaga_d <= FX2_FLAGA;
      end if;
   end process;

   process(IFCLK, RESET)
   begin
      if (RESET = '0') then
         if_cnt <= "101";
      elsif (IFCLK'event and IFCLK = '1') then
         if (FX2_FLAGA='0' and if_cnt="101") then
            if_cnt <= "101";
         else
            if_cnt <= if_cnt + '1';
         end if;
      end if;
   end process;

   process(IFCLK, RESET)
   begin
      if (RESET = '0') then
         slrd_uaa <= '1';
      elsif (IFCLK'event and IFCLK = '1') then
         if (if_cnt="000" and FX2_FLAGA='1') then
            slrd_uaa <= '0';
         elsif (if_cnt="100") then
            slrd_uaa <= '1';
         end if;
      end if;
   end process;

   process(IFCLK, RESET)
   begin
      if (RESET = '0') then
         add_mark_flg <= '0';
      elsif (IFCLK'event and IFCLK = '1') then
         if (FX2_FLAGA = '1' and flaga_d = '0') then
            add_mark_flg <= '1';
         elsif (if_cnt = "000") then
            add_mark_flg <= '0';
         end if;
      end if;
   end process;

   process(IFCLK)
   begin
      if (IFCLK'event and IFCLK = '1') then
         if (if_cnt = "000") then
            if (add_mark_flg = '1') then
               mark_fd <= '1';
            else
               mark_fd <= '0';
            end if;
         end if;
      end if;
   end process;

   process(IFCLK, RESET)
   begin
      if (RESET = '0') then
         ram_we <= '0';
      elsif (IFCLK'event and IFCLK = '1') then
         if (if_cnt="011" and slrd_uaa='0') then
            ram_we <= '1';
         else
            ram_we <= '0';
         end if;
      end if;
   end process;


   process(XIN, RESET)
   begin
      if (RESET = '0') then
         ram_re_mask <= '0';
      elsif (XIN'event and XIN = '1') then
         if (rd_data_count(10) = '1' and count = "111111111") then
            ram_re_mask <= '1';
         elsif (ram_re = '1') then
            if (rd_data_count = "00000000000") then
               ram_re_mask <= '0';
            end if;
         end if;
      end if;
   end process;

   process(XIN, RESET)
   begin
      if (RESET = '0') then
         ram_re <= '0';
      elsif (XIN'event and XIN = '1') then
         if (ram_re_mask = '0') then
            ram_re <= '0';
         else
            if (file_mode_2d(2 downto 1) = "01") then				-- PCM DOUBLE
               if (count(4 downto 0) = "00000") then
                  if ((count(6) and count(5)) = '0') then
                     ram_re <= '1';
                  end if;
               else
                  ram_re <= '0';
               end if;
            elsif (file_mode_2d(2 downto 1) = "00") then			-- PCM SINGLE
               if (count(5 downto 0) = "000000") then
                  if ((count(7) and count(6)) = '0') then
                     ram_re <= '1';
                  end if;
               else
                  ram_re <= '0';
               end if;
            else
               if (count(3 downto 0) = "0000") then
                  if ((count(5) and count(4)) = '0') then
                     ram_re <= '1';
                  end if;
               else
                  ram_re <= '0';
               end if;
            end if;
         end if;
      end if;
   end process;

   process(IFCLK, RESET)
   begin
      if (RESET = '0') then
         ram_waddr_reg_val <= '0';
      elsif (IFCLK'event and IFCLK = '1') then
         if (wr_data_count = "00000000000") then
            ram_waddr_reg_val <= '0';
         elsif (wr_data_count = "10000000000") then
            ram_waddr_reg_val <= '1';
         end if;
      end if;
   end process;

   process(IFCLK, RESET)
   begin
      if (RESET = '0') then
         ram_wr_addr_reg <= "10000000";
      elsif (IFCLK'event and IFCLK = '1') then
         if (ram_waddr_reg_val = '0') then
            ram_wr_addr_reg <= "10000000";
         else
            ram_wr_addr_reg <= wr_data_count(10 downto 3);
         end if;
      end if;
   end process;

   process(IFCLK)
   begin
      if (IFCLK'event and IFCLK = '1') then
         rd_rst_d <= rd_rst;
         rd_rst_2d <= rd_rst_d;
         rd_rst_3d <= rd_rst_2d;
      end if;
   end process;

   process(IFCLK, RESET)
   begin
      if (RESET = '0') then
         wr_rst <= '0';
      elsif (IFCLK'event and IFCLK = '1') then
         if (rd_rst_2d = '1' and rd_rst_3d = '0') then
            wr_rst <= '1';
         elsif (FX2_FLAGA = '1' and flaga_d = '0') then
            wr_rst <= '0';
         end if;
      end if;
   end process;

   process(IFCLK, RESET)
   begin
      if (RESET = '0') then
         file_mode <= "110";
      elsif (IFCLK'event and IFCLK = '1') then
         if (OUTLAT = '1') then
            file_mode <= DATAOUT(2 downto 0);
         end if;
      end if;
   end process;

   process(XIN)
   begin
      if (XIN'event and XIN = '1') then
         file_mode_d <= file_mode;
         file_mode_2d <= file_mode_d;
      end if;
   end process;

   process(XIN, RESET)
   begin
      if (RESET = '0') then
         count <= (others => '0');
      elsif (XIN'event and XIN = '1') then
         count <= count + '1';
      end if;
   end process;

   process(XIN)
   begin
      if (XIN'event and XIN = '1') then
         if (file_mode_2d(2 downto 1) = "01") then			-- PCM DOUBLE
            if (count(4 downto 0) = "10000") then
               if (ram_re_mask = '1') then
                  if (count(7 downto 5) = "000") then
                     pcm_datal(7 downto 0) <= pb_data;
                  elsif (count(7 downto 5) = "001") then
                     pcm_datal(15 downto 8) <= pb_data;
                  elsif (count(7 downto 5) = "010") then
                     pcm_datal(23 downto 16) <= pb_data;
                  end if;
               else
                  if (count(7 downto 5) = "000") then
                     pcm_datal <= (others => '0');
                  end if;
               end if;
            end if;
         elsif (file_mode_2d(2 downto 1) = "00") then			-- PCM SINGLE
            if (count(5 downto 0) = "100000") then
               if (ram_re_mask = '1') then
                  if (count(8 downto 6) = "000") then
                     pcm_datal(7 downto 0) <= pb_data;
                  elsif (count(8 downto 6) = "001") then
                     pcm_datal(15 downto 8) <= pb_data;
                  elsif (count(8 downto 6) = "010") then
                     pcm_datal(23 downto 16) <= pb_data;
                  end if;
               else
                  if (count(8 downto 6) = "000") then
                     pcm_datal <= (others => '0');
                  end if;
               end if;
            end if;
         else
            pcm_datal <= (others => '0');
         end if;
      end if;
   end process;

   process(XIN)
   begin
      if (XIN'event and XIN = '1') then
         if (file_mode_2d(2 downto 1) = "01") then			-- PCM DOUBLE
            if (count(4 downto 0) = "10000") then
               if (ram_re_mask = '1') then
                  if (count(7 downto 5) = "100") then
                     pcm_datar(7 downto 0) <= pb_data;
                  elsif (count(7 downto 5) = "101") then
                     pcm_datar(15 downto 8) <= pb_data;
                  elsif (count(7 downto 5) = "110") then
                     pcm_datar(23 downto 16) <= pb_data;
                  end if;
               else
                  if (count(7 downto 5) = "100") then
                     pcm_datar <= (others => '0');
                  end if;
               end if;
            end if;
         elsif (file_mode_2d(2 downto 1) = "00") then			-- PCM SINGLE
            if (count(5 downto 0) = "100000") then
               if (ram_re_mask = '1') then
                  if (count(8 downto 6) = "100") then
                     pcm_datar(7 downto 0) <= pb_data;
                  elsif (count(8 downto 6) = "101") then
                     pcm_datar(15 downto 8) <= pb_data;
                  elsif (count(8 downto 6) = "110") then
                     pcm_datar(23 downto 16) <= pb_data;
                  end if;
               else
                  if (count(8 downto 6) = "100") then
                     pcm_datar <= (others => '0');
                  end if;
               end if;
            end if;
         else
            pcm_datar <= (others => '0');
         end if;
      end if;
   end process;

   process(XIN, RESET)
   begin
      if (RESET = '0') then
         det_mark_err <= '0';
      elsif (XIN'event and XIN = '1') then
         if (file_mode_2d(2 downto 1) = "01") then			-- PCM DOUBLE
            if (count(4 downto 0) = "10000") then
               if (count(7 downto 5) = "001") then
                  det_mark_err <= mark;
               elsif (count(7 downto 5) = "010") then
                  det_mark_err <= mark;
               elsif (count(7 downto 5) = "100") then
                  det_mark_err <= mark;
               elsif (count(7 downto 5) = "101") then
                  det_mark_err <= mark;
               elsif (count(7 downto 5) = "110") then
                  det_mark_err <= mark;
               end if;
            else
               det_mark_err <= '0';
            end if;
         elsif (file_mode_2d(2 downto 1) = "00") then			-- PCM SINGLE
            if (count(5 downto 0) = "100000") then
               if (count(8 downto 6) = "001") then
                  det_mark_err <= mark;
               elsif (count(8 downto 6) = "010") then
                  det_mark_err <= mark;
               elsif (count(8 downto 6) = "100") then
                  det_mark_err <= mark;
               elsif (count(8 downto 6) = "101") then
                  det_mark_err <= mark;
               elsif (count(8 downto 6) = "110") then
                  det_mark_err <= mark;
               end if;
            else
               det_mark_err <= '0';
            end if;
         else
            det_mark_err <= '0';
         end if;
      end if;
   end process;

   process(XIN, RESET)
   begin
      if (RESET = '0') then
         rd_rst <= '0';
      elsif (XIN'event and XIN = '1') then
         if (det_mark_err = '1') then
            rd_rst <= '1';		-----> rd_data_count <= "00000000000"
         elsif (ram_re = '1') then
            rd_rst <= '0';		-----> ram_re_mask also becomes '0'
         end if;
      end if;
   end process;

   process(XIN)
   begin
      if (XIN'event and XIN = '1') then
         if (file_mode_2d(2 downto 1) = "01") then	-- PCM DOUBLE
            if (count(7 downto 0) = "11111111") then
               spd_para_dt <= pcm_datal;
            elsif (count(7 downto 0) = "01111111") then
               spd_para_dt <= pcm_datar;
            end if;
         elsif (file_mode_2d(2 downto 1) = "00") then	-- PCM SINGLE
            if (count(8 downto 0) = "111111111") then
               spd_para_dt <= pcm_datal;
            elsif (count(8 downto 0) = "011111111") then
               spd_para_dt <= pcm_datar;
            end if;
         else
            spd_para_dt <= (others => '0');
         end if;
      end if;
   end process;

   process(XIN)
   begin
      if (XIN'event and XIN = '1') then
         if (file_mode_2d(2 downto 1) = "01") then		-- PCM DOUBLE
            if (count(1 downto 0) = "10") then
               usb_bclk <= '1';
            elsif (count(1 downto 0) = "00") then
               usb_bclk <= '0';
            end if;
         elsif (file_mode_2d(2 downto 1) = "00") then		-- PCM SINGLE
            if (count(2 downto 0) = "100") then
               usb_bclk <= '1';
            elsif (count(2 downto 0) = "000") then
               usb_bclk <= '0';
            end if;
         else
            usb_bclk <= '0';
         end if;
      end if;
   end process;

   process(XIN)
   begin
      if (XIN'event and XIN = '1') then
         if (file_mode_2d(2 downto 1) = "01") then			-- PCM DOUBLE
            if (count(7 downto 0) = "00011100") then
               usb_lrck <= '0';					-- LRCK = LRCK
            elsif (count(7 downto 0) = "10011100") then
               usb_lrck <= '1';
            end if;
            if (count(6 downto 0) = "0000000") then
               usb_sdata <= '0';
            elsif (count(6 downto 0) = "0100000") then
               usb_sdata <= spd_para_dt(23);		-- SDATA = SDATA
            elsif (count(6 downto 0) = "0100100") then
               usb_sdata <= spd_para_dt(22);
            elsif (count(6 downto 0) = "0101000") then
               usb_sdata <= spd_para_dt(21);
            elsif (count(6 downto 0) = "0101100") then
               usb_sdata <= spd_para_dt(20);
            elsif (count(6 downto 0) = "0110000") then
               usb_sdata <= spd_para_dt(19);
            elsif (count(6 downto 0) = "0110100") then
               usb_sdata <= spd_para_dt(18);
            elsif (count(6 downto 0) = "0111000") then
               usb_sdata <= spd_para_dt(17);
            elsif (count(6 downto 0) = "0111100") then
               usb_sdata <= spd_para_dt(16);
            elsif (count(6 downto 0) = "1000000") then
               usb_sdata <= spd_para_dt(15);
            elsif (count(6 downto 0) = "1000100") then
               usb_sdata <= spd_para_dt(14);
            elsif (count(6 downto 0) = "1001000") then
               usb_sdata <= spd_para_dt(13);
            elsif (count(6 downto 0) = "1001100") then
               usb_sdata <= spd_para_dt(12);
            elsif (count(6 downto 0) = "1010000") then
               usb_sdata <= spd_para_dt(11);
            elsif (count(6 downto 0) = "1010100") then
               usb_sdata <= spd_para_dt(10);
            elsif (count(6 downto 0) = "1011000") then
               usb_sdata <= spd_para_dt(9);
            elsif (count(6 downto 0) = "1011100") then
               usb_sdata <= spd_para_dt(8);
            elsif (count(6 downto 0) = "1100000") then
               usb_sdata <= spd_para_dt(7);
            elsif (count(6 downto 0) = "1100100") then
               usb_sdata <= spd_para_dt(6);
            elsif (count(6 downto 0) = "1101000") then
               usb_sdata <= spd_para_dt(5);
            elsif (count(6 downto 0) = "1101100") then
               usb_sdata <= spd_para_dt(4);
            elsif (count(6 downto 0) = "1110000") then
               usb_sdata <= spd_para_dt(3);
            elsif (count(6 downto 0) = "1110100") then
               usb_sdata <= spd_para_dt(2);
            elsif (count(6 downto 0) = "1111000") then
               usb_sdata <= spd_para_dt(1);
            elsif (count(6 downto 0) = "1111100") then
               usb_sdata <= spd_para_dt(0);
            end if;
         elsif (file_mode_2d(2 downto 1) = "00") then			-- PCM SINGLE
            if (count = "000111000") then
               usb_lrck <= '0';					-- LRCK = LRCK
            elsif (count = "100111000") then
               usb_lrck <= '1';
            end if;
            if (count(7 downto 0) = "00000000") then
               usb_sdata <= '0';
            elsif (count(7 downto 0) = "01000000") then
               usb_sdata <= spd_para_dt(23);		-- SDATA = SDATA
            elsif (count(7 downto 0) = "01001000") then
               usb_sdata <= spd_para_dt(22);
            elsif (count(7 downto 0) = "01010000") then
               usb_sdata <= spd_para_dt(21);
            elsif (count(7 downto 0) = "01011000") then
               usb_sdata <= spd_para_dt(20);
            elsif (count(7 downto 0) = "01100000") then
               usb_sdata <= spd_para_dt(19);
            elsif (count(7 downto 0) = "01101000") then
               usb_sdata <= spd_para_dt(18);
            elsif (count(7 downto 0) = "01110000") then
               usb_sdata <= spd_para_dt(17);
            elsif (count(7 downto 0) = "01111000") then
               usb_sdata <= spd_para_dt(16);
            elsif (count(7 downto 0) = "10000000") then
               usb_sdata <= spd_para_dt(15);
            elsif (count(7 downto 0) = "10001000") then
               usb_sdata <= spd_para_dt(14);
            elsif (count(7 downto 0) = "10010000") then
               usb_sdata <= spd_para_dt(13);
            elsif (count(7 downto 0) = "10011000") then
               usb_sdata <= spd_para_dt(12);
            elsif (count(7 downto 0) = "10100000") then
               usb_sdata <= spd_para_dt(11);
            elsif (count(7 downto 0) = "10101000") then
               usb_sdata <= spd_para_dt(10);
            elsif (count(7 downto 0) = "10110000") then
               usb_sdata <= spd_para_dt(9);
            elsif (count(7 downto 0) = "10111000") then
               usb_sdata <= spd_para_dt(8);
            elsif (count(7 downto 0) = "11000000") then
               usb_sdata <= spd_para_dt(7);
            elsif (count(7 downto 0) = "11001000") then
               usb_sdata <= spd_para_dt(6);
            elsif (count(7 downto 0) = "11010000") then
               usb_sdata <= spd_para_dt(5);
            elsif (count(7 downto 0) = "11011000") then
               usb_sdata <= spd_para_dt(4);
            elsif (count(7 downto 0) = "11100000") then
               usb_sdata <= spd_para_dt(3);
            elsif (count(7 downto 0) = "11101000") then
               usb_sdata <= spd_para_dt(2);
            elsif (count(7 downto 0) = "11110000") then
               usb_sdata <= spd_para_dt(1);
            elsif (count(7 downto 0) = "11111000") then
               usb_sdata <= spd_para_dt(0);
            end if;
         else
            usb_lrck <= '0';
            usb_sdata <= '0';
         end if;
      end if;
   end process;

end rtl;
