library IEEE;
use IEEE.std_logic_1164.ALL;
use IEEE.std_logic_unsigned.ALL;

entity I2C_SLAVE is
   port (
    RESET       : in std_logic;
    SCLK        : in std_logic;
    SCL         : in std_logic;
    SDAI        : in std_logic;
    SDAO        : out std_logic;
    CS_N        : out std_logic;
    CCLK        : out std_logic;
    CDTI        : out std_logic;
    CDTO        : in std_logic;
    FPGA_IRQ    : out std_logic;
    USB_H       : in std_logic;		-- = VBUS_DET_H
    FILE_MODE   : in std_logic_vector(2 downto 0)
    );
end I2C_SLAVE;

architecture RTL of I2C_SLAVE is

constant   slvadr      : std_logic_vector (6 downto 0) := "1101011";

signal   Bit_count     : std_logic_vector (3 downto 0);
signal   getadr        : std_logic_vector (9 downto 0);
signal   sda_int_d     : std_logic;
signal   sda_int_2d    : std_logic;
signal   sda_int_3d    : std_logic;
signal   scl_int_d     : std_logic;
signal   scl_int_2d    : std_logic;
signal   scl_int_3d    : std_logic;
signal   is_scl_rise   : std_logic;
signal   is_scl_fall   : std_logic;
signal   is_sda_rise   : std_logic;
signal   is_sda_fall   : std_logic;
signal   ptr           : std_logic_vector (8 downto 0);
signal   tmp_ptr       : std_logic_vector (8 downto 0);
signal   tmp           : std_logic_vector (8 downto 0);
signal   is_write_z    : std_logic;
signal   kick_w        : std_logic;
signal   kick_r        : std_logic;
signal   restart_flag  : std_logic;
signal   count8        : std_logic_vector (7 downto 0);
signal   addr_int      : std_logic_vector (8 downto 0);
signal   data_int      : std_logic_vector (7 downto 0);
signal   rdbk_int      : std_logic_vector (7 downto 0);
signal   rdbk_reg      : std_logic_vector (7 downto 0);
signal   file_mode_d   : std_logic_vector(2 downto 0);
signal   file_mode_2d  : std_logic_vector(2 downto 0);
signal   file_mode_3d  : std_logic_vector(2 downto 0);
signal   count         : std_logic_vector(15 downto 0);
signal   usb_h_d       : std_logic;
signal   usb_h_2d      : std_logic;
signal   usb_h_3d      : std_logic;
signal   fs_irq        : std_logic;
signal   det_irq       : std_logic;	-- USB / SPDIF

BEGIN

    process(SCLK, RESET)
    begin
       if (RESET='0') then
          count <= ( others => '0' );
       elsif (SCLK' event and SCLK='1' ) then
          count <= count + '1';
       end if;
    end process;

    process(SCLK)
    begin
       if (SCLK' event and SCLK='1' ) then
          sda_int_d <= SDAI;
       end if;
    end process;

    process(SCLK)
    begin
       if (SCLK' event and SCLK='1' ) then
          sda_int_2d <= sda_int_d;
          sda_int_3d <= sda_int_2d;
       end if;
    end process;

    process(SCLK)
    begin
       if (SCLK' event and SCLK='1' ) then
          scl_int_d <= SCL;
       end if;
    end process;

    process(SCLK)
    begin
       if (SCLK' event and SCLK='1' ) then
          scl_int_2d <= scl_int_d;
          scl_int_3d <= scl_int_2d;
       end if;
    end process;

    process(SCLK, RESET)
    begin
       if (RESET = '0') then
          Bit_count <= "1000";
       elsif (SCLK' event and SCLK='1' ) then
          if (is_sda_fall = '1' and scl_int_3d = '1' and is_scl_rise = '0' and is_scl_fall = '0') then		-- is START condition
             Bit_count <= "1001";
          elsif (is_scl_fall = '1') then
             if (Bit_count = "0000") then
                Bit_count <= "1000";
             else
                Bit_count <= Bit_count - '1';
             end if;
          end if;
       end if;
    end process;

    is_scl_rise <= '1' when scl_int_2d = '1' and scl_int_3d = '0' else '0';
    is_scl_fall <= '1' when scl_int_2d = '0' and scl_int_3d = '1' else '0';
    is_sda_rise <= '1' when sda_int_2d = '1' and sda_int_3d = '0' else '0';
    is_sda_fall <= '1' when sda_int_2d = '0' and sda_int_3d = '1' else '0';

    process(SCLK)
    begin
       if (SCLK' event and SCLK='1' ) then
          if (kick_w = '1' or kick_r = '1') then
             addr_int <= ptr;
          end if;
       end if;
    end process;

    process(SCLK)
    begin
       if (SCLK' event and SCLK='1' ) then
          if (kick_w = '1') then
             data_int <= tmp(8 downto 1);
          end if;
       end if;
    end process;

    process(SCLK, RESET)
    variable CURRENT_STATE : std_logic_vector (2 downto 0);
    variable NEXT_STATE : std_logic_vector (2 downto 0);
    variable ADRS_count : std_logic;
    variable SLAV_hit   : std_logic;
    constant IDLE       : std_logic_vector (2 downto 0) := "000";
    constant SLAD       : std_logic_vector (2 downto 0) := "001";
    constant ADRSW      : std_logic_vector (2 downto 0) := "010";
    constant DATAW      : std_logic_vector (2 downto 0) := "011";
    constant DATAR      : std_logic_vector (2 downto 0) := "100";
    begin
       if (RESET='0') then
          CURRENT_STATE := IDLE;
          NEXT_STATE := IDLE;
          ADRS_count := '1';
          SLAV_hit := '0';
          SDAO <= '1';
          kick_w <= '0';
          kick_r <= '0';
          restart_flag <= '0';
          is_write_z <= '0';
       elsif (SCLK' event and SCLK='1' ) then
          CURRENT_STATE := NEXT_STATE;
          case CURRENT_STATE is
          when IDLE =>
             SDAO <= '1';
             if (is_sda_fall = '1' and scl_int_3d = '1' and is_scl_rise = '0' and is_scl_fall = '0') then		-- is START condition
                NEXT_STATE := SLAD;
             else
                NEXT_STATE := IDLE;
             end if;
          when SLAD =>
             if (is_scl_fall = '1') then
                getadr(conv_integer(Bit_count)) <= sda_int_3d;
                if (Bit_count = "1000") then
                   if (restart_flag = '1') then			-- next is read
                      kick_r <= '1';
                      is_write_z <= '1';
                   end if;
                elsif (Bit_count = "0001") then
                   if (getadr(8 downto 2) = slvadr) then		-- slave address hit
                      SLAV_hit := '1';
                      SDAO <= '0';					-- asert ack
                      rdbk_reg <= rdbk_int;
                   end if;
                elsif (Bit_count = "0000") then
                   if (SLAV_hit = '1') then
                      if (getadr(1) = '0') then				-- write
                         SDAO <= '1';					-- negate ack if aserted
                         NEXT_STATE := ADRSW;
                      else						-- read
                         SDAO <= rdbk_reg(7);
                         restart_flag <= '0';
                         NEXT_STATE := DATAR;
                      end if;
                      SLAV_hit := '0';
                   else
                      SDAO <= '1';					-- negate ack if aserted
                      restart_flag <= '0';
                      NEXT_STATE := IDLE;
                   end if;
                end if;
             else
                kick_r <= '0';
             end if;
          when ADRSW =>
             if (is_scl_fall = '1') then
                tmp_ptr(conv_integer(Bit_count)) <= sda_int_3d;
                if (Bit_count = "0001") then
                   SDAO <= '0';						-- asert ack
                elsif (Bit_count = "0000") then
                   SDAO <= '1';						-- negate ack
                   if (ADRS_count = '1') then
                      ptr(8) <= tmp_ptr(1);
                   else
                      ptr(7 downto 0) <= tmp_ptr(8 downto 1);
                      NEXT_STATE := DATAW;
                   end if;
                   ADRS_count := not ADRS_count;
                end if;
             end if;
          when DATAW =>
             if (is_sda_fall = '1' and scl_int_3d = '1' and is_scl_rise = '0' and is_scl_fall = '0') then		-- is START condition
                NEXT_STATE := SLAD;
                restart_flag <= '1';
             elsif (is_sda_rise = '1' and scl_int_3d = '1' and is_scl_rise = '0' and is_scl_fall = '0') then		-- is STOP condition
                NEXT_STATE := IDLE;
             elsif (is_scl_fall = '1') then
                tmp(conv_integer(Bit_count)) <= sda_int_3d;
                if (Bit_count = "0001") then
                   SDAO <= '0';						-- asert ack
                elsif (Bit_count = "0000") then
                   ptr <= ptr + '1';
                   SDAO <= '1';						-- negate ack
                end if;
             elsif (is_scl_rise = '1') then
                if (Bit_count = "0000") then
                   kick_w <= '1';
                   is_write_z <= '0';
                end if;
             else
                kick_w <= '0';
             end if;
          when DATAR =>
             if (is_scl_fall = '1') then
                if (Bit_count = "1000") then
                   SDAO <= rdbk_reg(6);
                   kick_r <= '1';
                   is_write_z <= '1';
                   ptr <= ptr + '1';
                elsif (Bit_count = "0111") then
                   SDAO <= rdbk_reg(5);
                elsif (Bit_count = "0110") then
                   SDAO <= rdbk_reg(4);
                elsif (Bit_count = "0101") then
                   SDAO <= rdbk_reg(3);
                elsif (Bit_count = "0100") then
                   SDAO <= rdbk_reg(2);
                elsif (Bit_count = "0011") then
                   SDAO <= rdbk_reg(1);
                elsif (Bit_count = "0010") then
                   SDAO <= rdbk_reg(0);
                elsif (Bit_count = "0001") then
                   rdbk_reg <= rdbk_int;
                   SDAO <= '1';
                elsif (Bit_count = "0000") then
                   SDAO <= rdbk_reg(7);
                   if (sda_int_3d = '1') then
                      NEXT_STATE := IDLE;
                   end if;
                else
                   SDAO <= '1';
                end if;
             else
                kick_r <= '0';
             end if;
          when others => null;
          end case;
       end if;
    end process;

    process(SCLK, RESET)
    begin
       if (RESET = '0') then
          count8 <= "11111111";
       elsif (SCLK' event and SCLK='1' ) then
          if (kick_w = '1' or kick_r = '1') then
             count8 <= "00000000";
          elsif (count8 = "11111111") then
             count8 <= "11111111";
          else
             count8 <= count8 + '1';
          end if;
       end if;
    end process;

   process(SCLK, RESET)
   begin
      if (RESET = '0') then
         CS_N <= '1';
      elsif (SCLK'event and SCLK = '1') then
         if (count8 = "00000000") then
            if ((ptr(8) or ptr(7) or ptr(6)) = '0') then
               CS_N <= '0';
            end if;
         elsif (count8 = "10000000") then
            CS_N <= '1';
         end if;
      end if;
   end process;

   process(SCLK, RESET)
   begin
      if (RESET = '0') then
         CCLK <= '1';
      elsif (SCLK'event and SCLK = '1') then
         if (count8(7) = '0') then
            if (count8(2 downto 0) = "000") then
               CCLK <= '0';
            elsif (count8(2 downto 0) = "100") then
               CCLK <= '1';
            end if;
         end if;
      end if;
   end process;

   process(SCLK, RESET)
   begin
      if (RESET = '0') then
         CDTI <= '0';
      elsif (SCLK'event and SCLK = '1') then
         if (count8(7) = '0') then
            if (count8(6 downto 0) = "0000000") then
               CDTI <= '0';
            elsif (count8(6 downto 0) = "0001000") then
               CDTI <= addr_int(5);
            elsif (count8(6 downto 0) = "0010000") then
               CDTI <= not is_write_z;
            elsif (count8(6 downto 0) = "0011000") then
               CDTI <= addr_int(4);
            elsif (count8(6 downto 0) = "0100000") then
               CDTI <= addr_int(3);
            elsif (count8(6 downto 0) = "0101000") then
               CDTI <= addr_int(2);
            elsif (count8(6 downto 0) = "0110000") then
               CDTI <= addr_int(1);
            elsif (count8(6 downto 0) = "0111000") then
               CDTI <= addr_int(0);
            elsif (count8(6 downto 0) = "1000000") then
               CDTI <= data_int(7);
            elsif (count8(6 downto 0) = "1001000") then
               CDTI <= data_int(6);
            elsif (count8(6 downto 0) = "1010000") then
               CDTI <= data_int(5);
            elsif (count8(6 downto 0) = "1011000") then
               CDTI <= data_int(4);
            elsif (count8(6 downto 0) = "1100000") then
               CDTI <= data_int(3);
            elsif (count8(6 downto 0) = "1101000") then
               CDTI <= data_int(2);
            elsif (count8(6 downto 0) = "1110000") then
               CDTI <= data_int(1);
            elsif (count8(6 downto 0) = "1111000") then
               CDTI <= data_int(0);
            end if;
         end if;
      end if;
   end process;

   process(SCLK)
   begin
      if (SCLK'event and SCLK = '1') then
         if (is_write_z = '1') then
            if ((ptr(8) or ptr(7) or ptr(6)) = '0') then
               if (count8 = "01000100") then
                  rdbk_int(7) <= CDTO;
               elsif (count8 = "01001100") then
                  rdbk_int(6) <= CDTO;
               elsif (count8 = "01010100") then
                  rdbk_int(5) <= CDTO;
               elsif (count8 = "01011100") then
                  rdbk_int(4) <= CDTO;
               elsif (count8 = "01100100") then
                  rdbk_int(3) <= CDTO;
               elsif (count8 = "01101100") then
                  rdbk_int(2) <= CDTO;
               elsif (count8 = "01110100") then
                  rdbk_int(1) <= CDTO;
               elsif (count8 = "01111100") then
                  rdbk_int(0) <= CDTO;
               end if;
            elsif (ptr = "100000000") then
               if (count8 = "01111100") then
                  rdbk_int(7) <= usb_h_3d;
                  rdbk_int(6 downto 0) <= "0000000";
               end if;
            elsif (ptr = "100000001") then
               if (count8 = "01111100") then
                  rdbk_int(7 downto 3) <= "00000";
                  rdbk_int(2 downto 0) <= FILE_MODE;
               end if;
            end if;
         end if;
      end if;
   end process;

   process(SCLK)
   begin
      if (SCLK' event and SCLK='1' ) then
         file_mode_d <= FILE_MODE;
         file_mode_2d <= file_mode_d;
         file_mode_3d <= file_mode_2d;
      end if;
   end process;

   process(SCLK)
   begin
      if (SCLK'event and SCLK = '1') then
         if (count = "0000000000000000") then
            usb_h_d <= USB_H;
            usb_h_2d <= usb_h_d;
         end if;
      end if;
   end process;

   process(SCLK)
   begin
      if (SCLK'event and SCLK = '1') then
         usb_h_3d <= usb_h_2d;
      end if;
   end process;

   process(SCLK, RESET)
   begin
      if (RESET = '0') then
         fs_irq <= '0';
      elsif (SCLK'event and SCLK = '1') then
         if (file_mode_d /= file_mode_2d) then
            fs_irq <= '1';
         elsif (kick_r = '1') then
            if (ptr = "100000001") then
               fs_irq <= '0';
            end if;
         end if;
      end if;
   end process;

   process(SCLK, RESET)
   begin
      if (RESET = '0') then
         det_irq <= '0';
      elsif (SCLK'event and SCLK = '1') then
         if (usb_h_2d /= usb_h_3d) then
            det_irq <= '1';
         elsif (kick_r = '1') then
            if (ptr = "100000000") then
               det_irq <= '0';
            end if;
         end if;
      end if;
   end process;

   FPGA_IRQ <= fs_irq or det_irq;

end RTL;
